<?php
/* --------------------------------------------------------------
 LoadUserPreferencesFromSession.php 2020-02-26
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Application\Kernel\Bootstrapper;

use Doctrine\DBAL\Connection;
use Gambio\Core\Configuration\ConfigurationService;
use Gambio\Core\Application\Kernel\Bootstrapper\Language\LanguageFinder;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Application\Application;
use Gambio\Core\Application\Kernel\AbstractBootstrapper;

/**
 * Class LoadUserPreferencesFromSession
 * @package Gambio\Core\Application\Kernel\Bootstrapper
 */
class LoadUserPreferencesFromSession extends AbstractBootstrapper
{
    /**
     * @inheritDoc
     */
    public function boot(Application $application): void
    {
        if (null === $_SESSION) {
            $newLanguage = $_GET['language'] ?? null;
            $finder      = $this->getLanguageFinder($application);
            $language    = $finder->load($newLanguage);
            
            $application->share(UserPreferences::class)->addArgument(null)->addArgument((int)$language->id());
            
            return;
        }
        
        if (array_key_exists('language', $_GET) || !array_key_exists('language', $_SESSION)) {
            $newLanguage = $_GET['language'] ?? null;
            $finder      = $this->getLanguageFinder($application);
            $language    = $finder->load($newLanguage);
            
            $_SESSION['language']         = $language->directory();
            $_SESSION['languages_id']     = $language->id();
            $_SESSION['language_charset'] = $language->charset();
            $_SESSION['language_code']    = $language->code();
        }
        
        $customerId = $this->getSessionValue('customer_id');
        $languageId = $this->getSessionValue('languages_id');
        
        $application->share(UserPreferences::class)->addArgument($customerId)->addArgument($languageId);
    }
    
    
    private function getLanguageFinder(Application $application): LanguageFinder
    {
        $configurationService = $application->get(ConfigurationService::class);
        $connection           = $application->get(Connection::class);
        
        return new LanguageFinder($configurationService, $connection);
    }
    
    
    /**
     * Returns the session value of given key, if available and null otherwise.
     *
     * @param string $key
     *
     * @return int|null
     */
    private function getSessionValue(string $key): ?int
    {
        return array_key_exists($key, $_SESSION) ? (int)$_SESSION[$key] : null;
    }
}